********************************************************************************
** ReplicationIdeology.do ********* November 21, 2024 **************************
** 
** Ecker-Ehrhardt, Matthias, Lisa Dellmuth, and Jonas Tallberg. 2024. 
** Ideology and Legitimacy in Global Governance.
** International Organization X:XX, p XXX-XXX
**
** Input: 
** 		WVS_Cross-national_Wave_7_Stata_v2_0.dta
**		WVS_trends_3_0_reduced.dta
**		Replication.dta
**
** Machine: Macbook Pro 16, Sequoia 15.1.1
** Program Version: Stata 17.0
**
********************************************************************************
********************************************************************************

* ssc install estout
* ssc install blindschemes
* ssc install coefplot

********************************************************************************
********************************************************************************
** Section "Puzzle" ************************************************************
********************************************************************************
********************************************************************************


use "WVS_Cross-national_Wave_7_Stata_v2_0.dta", clear


* weighting (post-stratification weights + 1000-equilibrated weight)

svyset [pweight=S018], strata(B_COUNTRY)

rename B_COUNTRY country
tab country

* code confidence variables

tab Q88
tab Q88, nol

label drop Q83 Q84 Q85 Q86 Q87 Q88 Q89

rename Q88 confWHO
rename Q89 confWTO
rename Q87 confWB
rename Q83 confUN
rename Q84 confIMF
rename Q85 confICC

recode confWTO  -5/-1=. 4=0 3=1 1=3
recode confWHO  -5/-1=. 4=0 3=1 1=3
recode confIMF -5/-1=. 4=0 3=1 1=3
recode confUN -5/-1=. 4=0 3=1 1=3
recode confWB -5/-1=. 4=0 3=1 1=3
recode confICC -5/-1=. 4=0 3=1 1=3

sum confUN confIMF confICC confWB confWHO confWTO
gen confios= (confUN +confIMF +confICC +confWB +confWHO +confWTO)/6

* IVs: socioeconomic status

tab Q275
tab Q275, nol
label drop Q275
rename Q275 education
recode education -5/-1=.
tab education

tab Q50
tab Q50, nol
label drop Q50
rename Q50 finsathousehold
recode finsathousehold -5/-1=. 1=0 2=1 3=2 4=3 5=4 6=5 7=6 8=7 9=8 10=9
tab finsathousehold

* IVs: domestic institutional trust

tab Q71
tab Q71, nol
label drop Q71
rename Q71 confgov
recode confgov -5/-1=. 4=0 3=1 1=3
tab confgov

tab Q252
tab Q252, nol
label drop Q252
rename Q252 polsatisfaction
recode polsatisfaction -5/-1=. 1=0 2=1 3=2 4=3 5=4 6=5 7=6 8=7 9=8 10=9
tab polsatisfaction

* IVs: identification

tab Q257
tab Q257, nol
label drop Q257
rename Q257 feelcountry
recode feelcountry -5/-1=. 1=3 4=0 3=1
tab feelcountry

tab Q259
tab Q259, nol
label drop Q259
rename Q259 feelworld
recode feelworld -5/-1=. 1=3 4=0 3=1
tab feelworld


* IVs: political values

tab Q154
tab Q154, nol
gen ordernation1 = Q154
recode ordernation1 -5/-1=. 2=0 3=0 4=0
tab ordernation1, missing


tab Q155
tab Q155, nol
gen ordernation2 = Q155
recode ordernation2 -5/-1=. 2=0 3=0 4=0
tab ordernation2, missing

gen comb_ordernation = ordernation1+ordernation2
tab comb_ordernation
recode comb_ordernation 2=0 1=0 0=1 
// code 0 for people who prioritized order in the nation (first or second place), 
// code 1 is those who prioritized something else
tab comb_ordernation


tab Q182
tab Q182, nol
label drop Q182
rename Q182 homo
recode homo -5/-1=. 1=0 2=1 3=2 4=3 5=4 6=5 7=6 8=7 9=8 10=9
tab homo

tab Q185
tab Q185, nol
label drop Q185
rename Q185 divorce
recode divorce -5/-1=. 1=0 2=1 3=2 4=3 5=4 6=5 7=6 8=7 9=8 10=9
tab divorce

tab Q186
tab Q186, nol
label drop Q186
rename Q186 sexmarriage
recode sexmarriage -5/-1=. 1=0 2=1 3=2 4=3 5=4 6=5 7=6 8=7 9=8 10=9
tab sexmarriage

tab Q184
tab Q184, nol
label drop Q184
rename Q184 abort
recode abort -5/-1=. 1=0 2=1 3=2 4=3 5=4 6=5 7=6 8=7 9=8 10=9

gen ethliberal = (homo+divorce+sexmarriage+abort)/4

tab Q121
tab Q121, nol
label drop Q121
rename Q121 immigrants
recode immigrants -5/-1=. 1=0 2=1 3=2 4=3 5=4
tab immigrants

gen TAN_GAL = ((ethliberal/9)+(immigrants/4)+(comb_ordernation/2))
mean TAN_GAL
gen GAL_TAN=-3.6*TAN_GAL+10

tab Q240
tab Q240, nol
label drop Q240
rename Q240 leftright
recode leftright -5/-1=. 1=0 2=1 3=2 4=3 5=4 6=5 7=6 8=7 9=8 10=9
gen leftright1 = leftright+1
tab leftright1


* IVs: controls

sum Q262
recode Q262 -5/-1=.
gen age = Q262
sum age 

tab Q57
tab Q57, nol
rename Q57 gentrust 
recode gentrust -5/-1=. 2=0 
tab gentrust

tab Q260
label drop Q260
gen male = Q260 // 1= male; 2= female
recode male -5/-1=. 2=0  // 1= male; 0= female
tab male


********************************************************************************
** Likelihood-ratio tests of (country-specific) random coefficients ************
********************************************************************************

*random-intercept model
mixed confios education finsathousehold  leftright1  GAL_TAN ///
	feelworld feelcountry confgov polsatisfaction age i.male ///
	|| country:, covariance(unstructured) mle 
eststo ri 

*random-slope model, with random slope for LR 
mixed confios education finsathousehold  leftright1  GAL_TAN ///
	feelworld feelcountry confgov polsatisfaction age i.male ///
	|| country:  leftright1, covariance(unstructured) mle
eststo rslr 

*random-slope model, with random slope for TG 
mixed confios education finsathousehold  leftright1  GAL_TAN ///
	feelworld feelcountry confgov polsatisfaction age i.male ///
	|| country:  GAL_TAN, covariance(unstructured) mle
eststo rsgal

lrtest ri rslr   // χ2(2) = 730.98, p<.0001
lrtest ri rsgal  // χ2(2) = 487.03, p<.0001

********************************************************************************
** Figure 1a. Association between left-right and IO confidence (all)
********************************************************************************

graph drop _all
set scheme plotplain

mixed confios education finsathousehold  leftright1  GAL_TAN ///
	feelworld feelcountry confgov polsatisfaction age i.male ///
	|| country:  leftright1, covariance(unstructured) mle
eststo Fig1a
margins, at(leftright1=(1(1)10))
marginsplot, recast(line) plot1opts(lcolor(gs8)) ciopt(color(black%20)) ///
	recastci(rarea) title("") xtitle("Left-Right") ytitle("IO confidence") ///
	ylabel(0(1)2, gmin) scale(1.6)   // was 1.4
graph export Figure1a.pdf, replace


********************************************************************************
** Figure 1b. Association between left-right and IO confidence (four countries)
********************************************************************************

mixed confios  leftright1   GAL_TAN education finsathousehold  ///
	feelworld feelcountry confgov polsatisfaction age i.male  ///
	if C_COW_ALPHA=="USA" | C_COW_ALPHA=="GMY" ///
	| C_COW_ALPHA=="BRA" | C_COW_ALPHA=="INS" ///
	|| country:  leftright1 , covariance(unstructured) mle
eststo Fig1b

predict re*, reffects 
predict re_se*, reses 
des re* 
matrix list e(b)
gen predicted_value = _b[_cons] + _b[ leftright1]* leftright1 + re2 + re1* leftright1  
gen predicted_value_se = sqrt(re_se2^2 + (re_se1* leftright1)^2)
gen lower_ci = predicted_value - 1.96*predicted_value_se
gen upper_ci = predicted_value + 1.96*predicted_value_se
sort country leftright1

twoway (rarea lower_ci upper_ci leftright1, connect(ascending) fcolor(gs12) ///
	lwidth(none)) || /// 
	(line predicted_value leftright1, connect(ascending) lwidth(thin) ///
	lpattern(solid) lcolor(black)) if predicted_value~=., ///
	ytitle(IO confidence) xtitle(Left-Right) xlabel(1(1)10) ylabel(0(1)2, gmin) /// 
	by(country, note("") scale(1.6) legend(off)) // was 1.4
graph export Figure1b.pdf, replace


********************************************************************************
** Figure 2a. Association between GAL-TAN and IO confidence (all)
********************************************************************************

mixed confios education finsathousehold  leftright1  GAL_TAN feelworld ///
	feelcountry confgov polsatisfaction age i.male || country:  GAL_TAN, ///
	covariance(unstructured) mle
eststo Fig2a

margins, at(GAL_TAN=(1(1)10))
marginsplot, recast(line) plot1opts(lcolor(gs8)) ciopt(color(black%20)) ///
	recastci(rarea) title("") xtitle("GAL-TAN") ytitle("IO confidence") ///
	ylabel(0(1)2, gmin) scale(1.6)  // was 1.4
graph export Figure2a.pdf, replace

********************************************************************************
** Figure 2b. Association between GAL-TAN and IO confidence (four countries)
********************************************************************************

mixed confios education finsathousehold  leftright1  GAL_TAN   ///
	feelworld feelcountry confgov polsatisfaction age i.male  ///
	if C_COW_ALPHA=="USA" | C_COW_ALPHA=="GMY" ///
	| C_COW_ALPHA=="BRA" | C_COW_ALPHA=="INS" ///
	|| country:  GAL_TAN  , covariance(unstructured) mle
eststo Fig2b

drop re1 re2 re_se* predicted_value predicted_value_se lower_ci upper_ci
predict re*, reffects 
predict re_se*, reses 
des re*
matrix list e(b)
gen predicted_value = _b[_cons] + _b[ GAL_TAN]* GAL_TAN + re2 + re1* GAL_TAN
gen predicted_value_se = sqrt(re_se2^2 + (re_se1* GAL_TAN)^2)
gen lower_ci = predicted_value - 1.96*predicted_value_se
gen upper_ci = predicted_value + 1.96*predicted_value_se
sort country GAL_TAN

twoway (rarea lower_ci upper_ci GAL_TAN, connect(ascending) /// 
	fcolor(gs12) lwidth(none)) || /// 
	(line predicted_value GAL_TAN, connect(ascending) lwidth(thin) ///
	lpattern(solid) lcolor(black)) if predicted_value~=., ///
	ytitle(IO confidence) xtitle(GAL-TAN) xlabel(1(1)10) ylabel(0(1)2, gmin) ///
	by(country, note("") scale(1.6)  legend(off))   // was 1.4
	
graph export Figure2b.pdf, replace

********************************************************************************
********************************************************************************
** Appendix A
********************************************************************************
********************************************************************************


********************************************************************************
** Table A1 Random-coefficient models with country-specific random slopes 
********************************************************************************


esttab Fig1a Fig2a  using TableA1.rtf, se(%9.3f) b(%9.3f)   ///
    transform(ln*: exp(@) exp(@) at*: tanh(@) (1-tanh(@)^2)) ///
    bic scalars(ll N) varwidth(16) replace

	
** Table A2 Country-specific OLS regression models

reg confios education finsathousehold  leftright1  GAL_TAN feelworld ///
	feelcountry confgov polsatisfaction age i.male  ///
	if C_COW_ALPHA=="BRA"
eststo brazil

reg confios education finsathousehold  leftright1  GAL_TAN feelworld ///
	feelcountry confgov polsatisfaction age i.male  ///
	if C_COW_ALPHA=="GMY"
eststo germany

reg confios education finsathousehold  leftright1  GAL_TAN feelworld ///
	feelcountry confgov polsatisfaction age i.male  ///
	if C_COW_ALPHA=="INS"
eststo indonesia

reg confios education finsathousehold  leftright1  GAL_TAN feelworld ///
	feelcountry confgov polsatisfaction age i.male  ///
	if C_COW_ALPHA=="USA"
eststo usa

esttab brazil germany indonesia usa using TableA2.rtf, se(%9.3f) b(%9.3f) l ///
      scalars(N) ar2 varwidth(16) starlevels(  * 0.05 ** 0.01 *** 0.001) replace


********************************************************************************
** Table A3 Descriptive Statistics
********************************************************************************


mixed confios  leftright1   GAL_TAN education finsathousehold  ///
	feelworld feelcountry confgov polsatisfaction age i.male  ///
	|| country:  leftright1 , covariance(unstructured) mle

tab country if e(sample)==1
keep if e(sample)==1
	
estpost tabstat ///
	conf* leftright1 GAL_TAN education finsathousehold  ///
	feelworld feelcountry confgov polsatisfaction ///
	age male, columns(statistics) statistics(count mean min max sd)

esttab using TableA3.rtf, cells  /// 
("count(fmt(%9.0f)) min(fmt(%9.0f)) sd(fmt(%9.3f)) max(fmt(%9.0f)) mean(fmt(%9.3f))") /// 
	label varwidth(30) noobs replace
	

********************************************************************************
********************************************************************************
* Section "Empirical Results" **************************************************
********************************************************************************
********************************************************************************


graph drop _all

********************************************************************************
**Figure 3a: Average perception of IO profiles, pooled sample
********************************************************************************


use Replication, replace
statsby,  : ci means q39 
gen IO=1
gen ID=1
save UN1, replace

use Replication, replace
statsby,  : ci means q43 
gen IO=1
gen ID=2
save UN2, replace

use Replication, replace
statsby,  : ci means q47 
gen IO=2
gen ID=1
save WHO1, replace

use Replication, replace
statsby,  : ci means q51 
gen IO=2
gen ID=2
save WHO2, replace

use Replication, replace
statsby,  : ci means q55 
gen IO=3
gen ID=1
save WB1, replace

use Replication, replace
statsby,  : ci means q59 
gen IO=3
gen ID=2
save WB2, replace

use Replication, replace
statsby,  : ci means q63 
gen IO=4
gen ID=1
save IMF1, replace

use Replication, replace
statsby,  : ci means q67 
gen IO=4
gen ID=2
save IMF2, replace

clear 
append using UN1 UN2 WHO1 WHO2 WB1 WB2 IMF1 IMF2, gen(filenum)

scatter IO mean  if ID==1,  ylabel(1 "UN" 2"WHO" 3 "WB" 4 "IMF") ///
	yscale(range(0.5 4.5) reverse) ytitle("") ///
	xline(5) xscale(range(4 6)) xlabel(4(.2)6) ///
 	title("Left-Right") legend(off) name(Pooled_LR)  /// 
	|| rcap ub lb IO if ID==1 , horizontal

scatter IO mean  if ID==2,  ylabel(1 "UN" 2"WHO" 3 "WB" 4 "IMF") ///
	yscale(range(0.5 4.5) reverse) ytitle("") ///
	xline(5) xscale(range(4 6)) xlabel(4(.2)6) ///
 	title("GAL-TAN") legend(off) name(Pooled_GT)  /// 
	|| rcap ub lb IO if ID==2 , horizontal

gr combine ///
	Pooled_LR ///	
	Pooled_GT ///	
	, col(1) iscale(1.2) title("Pooled Data", size(large)) imargin(1 10 1 5) ysize(3.2) // was .8
	
graph export Figure3a.pdf, replace


********************************************************************************
**Figure 3b: Average perception of IO profiles, country-specific samples
********************************************************************************

use Replication, replace
statsby, by(ncountry2) : ci means q39
gen IO=1
gen ID=1
save UN1, replace

use Replication, replace
statsby, by(ncountry2) : ci means q43
gen IO=1
gen ID=2
save UN2, replace

use Replication, replace
statsby, by(ncountry2) : ci means q47
gen IO=2
gen ID=1
save WHO1, replace

use Replication, replace
statsby, by(ncountry2) : ci means q51
gen IO=2
gen ID=2
save WHO2, replace

use Replication, replace
statsby, by(ncountry2) : ci means q55
gen IO=3
gen ID=1
save WB1, replace

use Replication, replace
statsby, by(ncountry2) : ci means q59
gen IO=3
gen ID=2
save WB2, replace

use Replication, replace
statsby, by(ncountry2) : ci means q63
gen IO=4
gen ID=1
save IMF1, replace

use Replication, replace
statsby, by(ncountry2) : ci means q67
gen IO=4
gen ID=2
save IMF2, replace

clear 
append using UN1 UN2 WHO1 WHO2 WB1 WB2 IMF1 IMF2, gen(filenum)

gen country=" "
replace country="Brazil"        if ncountry2==1
replace country="Germany"       if ncountry2==2
replace country="Indonesia"     if ncountry2==3
replace country="United States" if ncountry2==4

foreach c in "Brazil" "Germany" "Indonesia" {

scatter IO mean  if ID==1 & country=="`c'",   /// 
ylabel(1 "UN" 2"WHO" 3 "WB" 4 "IMF") ///
	yscale(range(0.5 4.5) reverse) ytitle("") ///
	xline(5) xscale(range(4 6)) xlabel(4(0.2)6) ///
 	title("Left-Right") legend(off) name(Pooled_LR_`c')  /// 
	|| rcap ub lb IO if ID==1 & country=="`c'", horizontal

scatter IO mean  if ID==2 & country=="`c'",   /// 
ylabel(1 "UN" 2"WHO" 3 "WB" 4 "IMF") ///
	yscale(range(0.5 4.5) reverse) ytitle("") ///
	xline(5) xscale(range(4 6)) xlabel(4(0.2)6) ///
 	title("GAL-TAN") legend(off) name(Pooled_GT_`c')  /// 
	|| rcap ub lb IO if ID==2 & country=="`c'", horizontal

gr combine ///
	Pooled_LR_`c' ///	
	Pooled_GT_`c' ///	
	, col(1) iscale(.8) name("`c'") title("`c'") imargin(1 10 1 5)
}

scatter IO mean  if ID==1 & country=="United States",   /// 
ylabel(1 "UN" 2"WHO" 3 "WB" 4 "IMF") ///
	yscale(range(0.5 4.5) reverse) ytitle("") ///
	xline(5) xscale(range(4 6)) xlabel(4(0.2)6) ///
 	title("Left-Right") legend(off) name(Pooled_LR_US)  /// 
	|| rcap ub lb IO if ID==1 & country=="United States", horizontal

scatter IO mean  if ID==2 & country=="United States",   /// 
ylabel(1 "UN" 2"WHO" 3 "WB" 4 "IMF") ///
	yscale(range(0.5 4.5) reverse) ytitle("") ///
	xline(5) xscale(range(4 6)) xlabel(4(0.2)6) ///
 	title("GAL-TAN") legend(off) name(Pooled_GT_US)  /// 
	|| rcap ub lb IO if ID==2 & country=="United States", horizontal

gr combine ///
	Pooled_LR_US ///	
	Pooled_GT_US ///	
, col(1) iscale(.8) name("UnitedStates") title("United States") imargin(1 10 1 5)

**
	
gr combine ///
	Brazil ///	
	Germany ///	
	Indonesia ///
	UnitedStates ///
	, col(2) iscale(.75) imargin(0 0 0 6) ysize(5) // was .7 and ysize unspecified
	
graph export Figure3b.pdf, replace

erase UN1.dta 
erase UN2.dta 
erase WHO1.dta 
erase WHO2.dta 
erase WB1.dta 
erase WB2.dta 
erase IMF1.dta 
erase IMF2.dta


********************************************************************************
** Figure 4: Marginal effect of ideological orientation on IO confidence
********************************************************************************

use Replication, replace

vl create CONTROLS = (intcoop confgov finsathousehold polsatisfaction  /// 
	feelcountry feelworld age male education GE IN US)

estimates drop _all
graph drop _all
set scheme plotplain

** UN

reg confUN c.q3##c.q39 c.q17##c.q43 $CONTROLS
estimates store UN

margins, dydx(q3) at(q39=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  /// 
	size(small)) ylabel(-.4(.1).4, gmin) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(UN1) title("UN")
	
margins, dydx(q17) at(q43=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  /// 
	size(small)) ylabel(-.4(.1).4, gmin) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(UN2) title("UN")

	
** WHO

reg confWHO c.q3##c.q47 c.q17##c.q51 $CONTROLS 
estimates store WHO

margins, dydx(q3) at(q47=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  /// 
	size(small)) ylabel(-.4(.1).4, gmin) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(WHO1) title("WHO")

margins, dydx(q17) at(q51=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  /// 
	size(small)) ylabel(-.4(.1).4, gmin) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(WHO2) title("WHO")

** WB
	
reg confWB c.q3##c.q55 c.q17##c.q59 $CONTROLS 
estimates store WB

margins, dydx(q3) at(q55=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  /// 
	size(small)) ylabel(-.4(.1).4, gmin) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(WB1) title("WB")

margins, dydx(q17) at(q59=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  /// 
	size(small)) ylabel(-.4(.1).4, gmin) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(WB2) title("WB")


** IMF


reg confIMF c.q3##c.q63 c.q17##c.q67 $CONTROLS 
estimates store IMF

margins, dydx(q3) at(q63=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  /// 
	size(small)) ylabel(-.4(.1).4, gmin) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(IMF1) title("IMF")

margins, dydx(q17) at(q67=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  /// 
	size(small)) ylabel(-.4(.1).4, gmin) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(IMF2) title("IMF")


esttab * using TableC4.rtf, se(%9.3f) b(%9.3f) l ///
    ar2 scalars(N) varwidth(16) starlevels(  * 0.05 ** 0.01 *** 0.001) replace

gr combine ///
	UN1 ///
	WHO1 ///	
	WB1 ///
	IMF1 ///
	UN2 ///	
	WHO2 ///	
	WB2 ///
	IMF2 ///
	, col(4) iscale(.8) imargin(0 3 0 6) // was .65 and ysize unspecified
	
graph export Figure4.pdf, replace


********************************************************************************
** Figure 5: Marginal effects of ideological orientation on IO confidence
********************************************************************************

use Replication, replace
vl create CONTROLS = (intcoop confgov finsathousehold polsatisfaction  /// 
feelcountry feelworld age male education GE IN US)
estimates drop _all
graph drop _all
set scheme plotplain

**Migration 

gen treat1 = E1_34c
label value treat1 m_treatmentLRc
reg   q19 c.q3##b1.treat1 q17 $CONTROLS  if c_0034==q20  | c_0034==-66 //LR
estimates store E1_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  /// 
	size(small)) ylabel(-.2(.1).4, gmin gmax) yscale(titlegap(*-05)) ///
	xtitle("") xscale(range(-.2 2.2)) yline(0,  /// 
	lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E1_34c) title("Migration")

gen treat2=E1_12c
label value treat2 m_treatmentGTc
reg  q19 c.q17##b1.treat2 q3  $CONTROLS  if c_0034==q20 | c_0034==-66 //GAL-TAN
estimates store E1_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation", size(small))  /// 
	ylabel(-.2(.1).4, gmin gmax) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2)) yline(0,  /// 
	lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E1_12c) title("Migration")


**Trade 

replace treat1=E2_34c
label value treat1 m_treatmentLRc
reg   q22 c.q3##b1.treat1 q17 $CONTROLS  if c_0035==q23  | c_0035==-66 //LR
estimates store E2_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation", size(small))   /// 
	ylabel(-.2(.1).4, gmin gmax) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2)) yline(0,  /// 
	lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E2_34c) title("Trade")

replace treat2=E2_12c
label value treat2 m_treatmentGTc
reg  q22 c.q17##b1.treat2 q3 $CONTROLS  if c_0035==q23 | c_0035==-66 //GAL-TAN
estimates store E2_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation", size(small))   /// 
	ylabel(-.2(.1).4, gmin gmax) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2)) yline(0,  /// 
	lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E2_12c) title("Trade")


**Climate 

replace treat1=E3_34c
label value treat1 m_treatmentLRc
reg  q25 c.q3##b1.treat1 q17 $CONTROLS  if c_0036==q26  | c_0036==-66 //LR
estimates store E3_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  /// 
	size(small))  ylabel(-.2(.1).4, gmin gmax) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2)) yline(0,  /// 
	lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E3_34c) title("Climate")
	
replace treat2=E3_12c
label value treat2 m_treatmentGTc
reg  q25 c.q17##b1.treat2 q3 $CONTROLS  if c_0036==q26 | c_0036==-66 //GAL-TAN
estimates store E3_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  /// 
	size(small))  ylabel(-.2(.1).4, gmin gmax) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2)) yline(0,  /// 
	lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E3_12c) title("Climate")

	
		
**Peacebuilding 

replace treat1=E4_34c
label value treat1 m_treatmentLRc
reg  q28 c.q3##b1.treat1 q17 $CONTROLS  if c_0037==q29  | c_0037==-66 //LR
estimates store E4_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  /// 
	size(small))  ylabel(-.2(.1).4, gmin gmax) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2)) yline(0,  /// 
	lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E4_34c) title("Peace")

replace treat2=E4_12c
label value treat2 m_treatmentGTc
reg  q28 c.q17##b1.treat2 q3 $CONTROLS  if c_0037==q29 | c_0037==-66 //GAL-TAN
estimates store E4_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  /// 
	size(small))  ylabel(-.2(.1).4, gmin gmax) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2)) yline(0,  /// 
	lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E4_12c) title("Peace")


esttab * using TableD2.rtf, se(%9.3f) b(%9.3f) l nobase ///
      scalars(N) ar2 varwidth(16) starlevels(  * 0.05 ** 0.01 *** 0.001) replace
	  
gr combine ///
	E1_34c ///
	E2_34c ///
	E3_34c ///
	E4_34c ///
	E1_12c ///
	E2_12c ///	
	E3_12c ///
	E4_12c ///	
	, col(4)   iscale(.8) imargin(0 3 0 6) // was .65 and ysize unspecified
graph export Figure5.pdf, replace   




********************************************************************************
********************************************************************************
** Appendix B Survey Design and Sample Composition
********************************************************************************
********************************************************************************

use Replication, replace

estimates drop _all
graph drop _all
set scheme plotplain

********************************************************************************
** Table B2: Sample size
********************************************************************************

asdoc tab ncountry, save(TableB2.rtf) replace 

********************************************************************************
** Table B3a: Sample composition - gender
********************************************************************************

asdoc tab male ncountry, col nof save(TableB3a.rtf) replace 

********************************************************************************
** Table B4a: Sample composition - age
********************************************************************************

asdoc tab agec ncountry, col nof save(TableB2a.rtf) replace 

********************************************************************************
** Table B5a: Sample composition - education
********************************************************************************

asdoc tab education3 ncountry, col nof save(TableB5a.rtf) replace 




********************************************************************************
********************************************************************************
** Appendix C Observational Study
********************************************************************************
********************************************************************************

use Replication, replace

estimates drop _all
graph drop _all
set scheme plotplain

********************************************************************************
** Table C1: Descriptive statistics 
********************************************************************************

estpost tabstat ///
	confUN confWHO confWB  confIMF confgov q3 q17 q39 q47 q55 q63 q43 q51 q59 ///
	q67 q19 q22 q25 q28 feelworld feelcountry polsatisfaction finsathousehold ///
	age male education, columns(statistics) statistics(count mean min max sd)

esttab using TableC1.rtf, cells  /// 
("count(fmt(%9.0f)) min(fmt(%9.0f)) sd(fmt(%9.3f)) max(fmt(%9.0f)) mean(fmt(%9.3f))") /// 
	label varwidth(30) noobs replace


********************************************************************************
** Table C2: Bivariate associations between self-placement and IO-profiles
********************************************************************************

reg q3 q39  
estimates store UN1

reg q17 q43 
estimates store UN2

reg q3 q47 
estimates store WHO1

reg q17 q51 
estimates store WHO2
 
reg q3 q55 
estimates store WB1

reg q17 q59 
estimates store WB2

reg q3 q63 
estimates store IMF1

reg q17 q67 
estimates store IMF2

esttab * using TableC2.rtf, se(%9.3f) beta(%9.3f) l ///
    ar2 scalars(N) varwidth(16) starlevels(  * 0.05 ** 0.01 *** 0.001) replace



********************************************************************************
** Table C3a: Paired t-tests (two-tailed) of perceived IO profiles (pooled)
********************************************************************************

asdoc ttest q39==q47, save(TableC3a.rtf) replace 
asdoc ttest q39==q55, rowappend 
asdoc ttest q39==q63, rowappend 
asdoc ttest q47==q55, rowappend 
asdoc ttest q47==q63, rowappend 
asdoc ttest q55==q63, rowappend 

asdoc ttest q43==q51, rowappend   
asdoc ttest q43==q59, rowappend 
asdoc ttest q43==q67, rowappend 
asdoc ttest q51==q59, rowappend 
asdoc ttest q51==q67, rowappend 
asdoc ttest q59==q67, rowappend 

********************************************************************************
** Table C3b-e: Paired t-tests (two-tailed) of perceived IO profiles (countries)
********************************************************************************

foreach c in GE BR IN US {

asdoc ttest q39==q47 if `c'==1, save(TableC3_`c'.rtf) replace
asdoc ttest q39==q55 if `c'==1, rowappend 
asdoc ttest q39==q63 if `c'==1, rowappend 
asdoc ttest q47==q55 if `c'==1, rowappend 
asdoc ttest q47==q63 if `c'==1, rowappend 
asdoc ttest q55==q63 if `c'==1, rowappend 

asdoc ttest q43==q51 if `c'==1, rowappend   
asdoc ttest q43==q59 if `c'==1, rowappend 
asdoc ttest q43==q67 if `c'==1, rowappend 
asdoc ttest q51==q59 if `c'==1, rowappend 
asdoc ttest q51==q67 if `c'==1, rowappend 
asdoc ttest q59==q67 if `c'==1, rowappend 
}

********************************************************************************
** Table C4: OLS regression, as illustrated in Figure 3 in the paper
********************************************************************************

* See code for Figure 3 above


********************************************************************************
** Table C5: OLS regression, without IO profiles for comparison of adj. R2
********************************************************************************

use Replication, replace

vl create CONTROLS = (intcoop confgov finsathousehold polsatisfaction ///
	feelcountry feelworld age male education GE IN US)

estimates drop _all

** UN

reg confUN c.q3##c.q39 c.q17##c.q43 $CONTROLS

estimates store UN

reg confUN c.q3  c.q17  $CONTROLS if  q39~=. & q43~=.
estimates store UN0

lrtest UN UN0

** WHO

reg confWHO c.q3##c.q47 c.q17##c.q51 $CONTROLS 
estimates store WHO

reg confWHO c.q3  c.q17  $CONTROLS if  q47~=. & q51~=.
estimates store WHO0

lrtest WHO WHO0

** WB
	
reg confWB c.q3##c.q55 c.q17##c.q59 $CONTROLS 
estimates store WB

reg confWB c.q3  c.q17  $CONTROLS if q55~=. & q59~=.
estimates store WB0

lrtest WB WB0


** IMF
	
reg confIMF c.q3##c.q63 c.q17##c.q67 $CONTROLS   
estimates store IMF

reg confIMF c.q3  c.q17  $CONTROLS if  q63~=. & q67~=.
estimates store IMF0

lrtest IMF IMF0

esttab *0 using TableC5.rtf, se(%9.3f) b(%9.3f) l nobase ///
    ar2 scalars(N) varwidth(16) starlevels(  * 0.05 ** 0.01 *** 0.001) replace

	
********************************************************************************
** Table C6: OLS regression results in Table C4, replicated with weights 
********************************************************************************

use Replication, replace
vl create CONTROLS = (intcoop confgov finsathousehold polsatisfaction  /// 
feelcountry feelworld age male education GE IN US)
estimates drop _all
graph drop _all
set scheme plotplain


** UN

reg confUN c.q3##c.q39 c.q17##c.q43 $CONTROLS [pw=weight]
estimates store UN

margins, dydx(q3) at(q39=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(UN1) title("UN")
	
margins, dydx(q17) at(q43=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(UN2) title("UN")

	
** WHO

reg confWHO c.q3##c.q47 c.q17##c.q51 $CONTROLS [pw=weight]
estimates store WHO

margins, dydx(q3) at(q47=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(WHO1) title("WHO")

margins, dydx(q17) at(q51=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(WHO2) title("WHO")

** WB
	
reg confWB c.q3##c.q55 c.q17##c.q59 $CONTROLS [pw=weight]
estimates store WB

margins, dydx(q3) at(q55=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(WB1) title("WB")

margins, dydx(q17) at(q59=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(WB2) title("WB")


** IMF


reg confIMF c.q3##c.q63 c.q17##c.q67 $CONTROLS [pw=weight]
estimates store IMF

margins, dydx(q3) at(q63=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(IMF1) title("IMF")

margins, dydx(q17) at(q67=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(IMF2) title("IMF")

esttab * using TableC6.rtf, se(%9.3f) b(%9.3f) l nobase ///
    ar2 scalars(N) varwidth(16) starlevels(  * 0.05 ** 0.01 *** 0.001) replace

gr combine ///
	UN1 ///
	WHO1 ///	
	WB1 ///
	IMF1 ///
	UN2 ///	
	WHO2 ///	
	WB2 ///
	IMF2 ///
	, col(4) iscale(.8) imargin(0 3 0 6) 
graph export FigureC6.pdf, replace



********************************************************************************
** Table C7: OLS regression results in Table C4, one dimension at a time  
********************************************************************************

use Replication, replace
vl create CONTROLS = (intcoop confgov finsathousehold polsatisfaction  /// 
feelcountry feelworld age male education GE IN US)
estimates drop _all
graph drop _all
set scheme plotplain


** UN

reg confUN c.q3##c.q39  $CONTROLS [pw=weight]
estimates store UN1

margins, dydx(q3) at(q39=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(UN1) title("UN")

reg confUN  c.q17##c.q43 $CONTROLS [pw=weight]
estimates store UN2

margins, dydx(q17) at(q43=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(UN2) title("UN")

	
** WHO

reg confWHO c.q3##c.q47  $CONTROLS [pw=weight]
estimates store WHO1

margins, dydx(q3) at(q47=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(WHO1) title("WHO")

reg confWHO c.q17##c.q51 $CONTROLS [pw=weight]
estimates store WHO2

margins, dydx(q17) at(q51=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(WHO2) title("WHO")

** WB
	
reg confWB c.q3##c.q55 $CONTROLS [pw=weight]
estimates store WB1

margins, dydx(q3) at(q55=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(WB1) title("WB")

reg confWB  c.q17##c.q59 $CONTROLS [pw=weight]
estimates store WB2

margins, dydx(q17) at(q59=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(WB2) title("WB")


** IMF


reg confIMF c.q3##c.q63  $CONTROLS [pw=weight]
estimates store IMF1

margins, dydx(q3) at(q63=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(IMF1) title("IMF")

reg confIMF c.q17##c.q67 $CONTROLS [pw=weight]
estimates store IMF2

margins, dydx(q17) at(q67=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(IMF2) title("IMF")

esttab * using TableC7.rtf, se(%9.3f) b(%9.3f) l nobase ///
    ar2 scalars(N) varwidth(16) starlevels(  * 0.05 ** 0.01 *** 0.001) replace

gr combine ///
	UN1 ///
	WHO1 ///	
	WB1 ///
	IMF1 ///
	UN2 ///	
	WHO2 ///	
	WB2 ///
	IMF2 ///
	, col(4) iscale(.8) imargin(0 3 0 6) 
graph export FigureC7.pdf, replace



********************************************************************************
** Table C8: OLS regression results in Table C4, attentive respondents only
********************************************************************************

use Replication, replace
vl create CONTROLS = (intcoop confgov finsathousehold polsatisfaction  /// 
feelcountry feelworld age male education GE IN US)
estimates drop _all
graph drop _all
set scheme plotplain

keep if attention==1


** UN

reg confUN c.q3##c.q39 c.q17##c.q43 $CONTROLS
estimates store UN

margins, dydx(q3) at(q39=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(UN1) title("UN")
	
margins, dydx(q17) at(q43=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(UN2) title("UN")

	
** WHO

reg confWHO c.q3##c.q47 c.q17##c.q51 $CONTROLS 
estimates store WHO

margins, dydx(q3) at(q47=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(WHO1) title("WHO")

margins, dydx(q17) at(q51=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(WHO2) title("WHO")

** WB
	
reg confWB c.q3##c.q55 c.q17##c.q59 $CONTROLS 
estimates store WB

margins, dydx(q3) at(q55=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(WB1) title("WB")

margins, dydx(q17) at(q59=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(WB2) title("WB")


** IMF


reg confIMF c.q3##c.q63 c.q17##c.q67 $CONTROLS 
estimates store IMF

margins, dydx(q3) at(q63=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(IMF1) title("IMF")

margins, dydx(q17) at(q67=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(IMF2) title("IMF")

esttab * using TableC8.rtf, se(%9.3f) b(%9.3f) l nobase ///
    ar2 scalars(N) varwidth(16) starlevels(* 0.05 ** 0.01 *** 0.001) replace

gr combine ///
	UN1 ///
	WHO1 ///	
	WB1 ///
	IMF1 ///
	UN2 ///	
	WHO2 ///	
	WB2 ///
	IMF2 ///
	, col(4) iscale(.8) imargin(0 3 0 6) 
graph export FigureC8.pdf, replace



********************************************************************************
** Table C9-12: OLS regression, as Table C4/Figure 4 but for spec countries 
********************************************************************************


foreach c in BR GE IN US {

********

use Replication, replace
vl create CONTROLS = (intcoop confgov finsathousehold polsatisfaction  /// 
feelcountry feelworld age male education)
estimates drop _all
graph drop _all
set scheme plotplain

keep if `c'==1


** UN

reg confUN c.q3##c.q39 c.q17##c.q43 $CONTROLS
estimates store UN

margins, dydx(q3) at(q39=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(UN1) title("UN")
	
margins, dydx(q17) at(q43=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(UN2) title("UN")

	
** WHO

reg confWHO c.q3##c.q47 c.q17##c.q51 $CONTROLS 
estimates store WHO

margins, dydx(q3) at(q47=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(WHO1) title("WHO")

margins, dydx(q17) at(q51=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(WHO2) title("WHO")

** WB
	
reg confWB c.q3##c.q55 c.q17##c.q59 $CONTROLS 
estimates store WB

margins, dydx(q3) at(q55=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(WB1) title("WB")

margins, dydx(q17) at(q59=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(WB2) title("WB")


** IMF


reg confIMF c.q3##c.q63 c.q17##c.q67 $CONTROLS 
estimates store IMF

margins, dydx(q3) at(q63=(0(1)10))
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(IMF1) title("IMF")

margins, dydx(q17) at(q67=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(IMF2) title("IMF")


esttab * using TableC_`c'.rtf, se(%9.3f) b(%9.3f) l nobase ///
    ar2 scalars(N) varwidth(16) starlevels(  * 0.05 ** 0.01 *** 0.001) replace

gr combine ///
	UN1 ///
	WHO1 ///	
	WB1 ///
	IMF1 ///
	UN2 ///	
	WHO2 ///	
	WB2 ///
	IMF2 ///
	, col(4) iscale(.8) imargin(0 3 0 6) 
graph export FigureC_`c'.pdf, replace

*******
} 

********************************************************************************
** Table C13: OLS-Regression, pooled across IOs
********************************************************************************


* UN

use Replication, replace
estimates drop _all
graph drop _all
set scheme plotplain


rename confUN confIO
rename q39 IOlr
rename q43 IOgt

keep id GE BR IN US confIO IOlr IOgt q3 q17 ///
	intcoop confgov finsathousehold polsatisfaction ///
	feelcountry feelworld age male education weight attention

gen IO  = 1
gen UN  = 1
gen WHO = 0
gen WB  = 0
gen IMF = 0

save ReplicationPooled_1, replace

*WHO

use Replication, replace
estimates drop _all
graph drop _all
set scheme plotplain

rename confWHO confIO
rename q47 IOlr
rename q51 IOgt

keep id GE BR IN US confIO IOlr IOgt q3 q17 ///
	intcoop confgov finsathousehold polsatisfaction ///
	feelcountry feelworld age male education weight attention

gen IO  = 2
gen UN  = 0
gen WHO = 1
gen WB  = 0
gen IMF = 0

save ReplicationPooled_2, replace

*WB

use Replication, replace
estimates drop _all
graph drop _all
set scheme plotplain


rename confWB confIO
rename q55 IOlr
rename q59 IOgt

keep id GE BR IN US confIO IOlr IOgt q3 q17 ///
	intcoop confgov finsathousehold polsatisfaction ///
	feelcountry feelworld age male education weight attention

gen IO  = 3
gen UN  = 0
gen WHO = 0
gen WB  = 1
gen IMF = 0

save ReplicationPooled_3, replace


*IMF

use Replication, replace
estimates drop _all
graph drop _all
set scheme plotplain


rename confIMF confIO
rename q63 IOlr
rename q67 IOgt

keep id GE BR IN US confIO IOlr IOgt q3 q17 ///
	intcoop confgov finsathousehold polsatisfaction ///
	feelcountry feelworld age male education weight attention

gen IO  = 4
gen UN  = 0
gen WHO = 0
gen WB  = 0
gen IMF = 1

save ReplicationPooled_4, replace

*append

use ReplicationPooled_1, replace
append using ReplicationPooled_2
append using ReplicationPooled_3
append using ReplicationPooled_4

sum
tab IO
sort id

save ReplicationPooled, replace

* models ******************

use ReplicationPooled, replace
vl create CONTROLS = (intcoop confgov finsathousehold polsatisfaction feelcountry feelworld age male education WHO WB IMF)

estimates drop _all
graph drop _all
set scheme plotplain

reg confIO c.q3##c.IOlr c.q17##c.IOgt $CONTROLS GE IN US 
estimates store FullSample

margins, dydx(q3) at(IOlr=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation", size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(FullSample1) title("FullSample")

margins, dydx(q17) at(IOgt=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation", size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(FullSample2) title("FullSample")

	

foreach c in GE BR IN US {

********

use ReplicationPooled, replace
vl create CONTROLS = (intcoop confgov finsathousehold polsatisfaction feelcountry feelworld age male education WHO WB IMF)

keep if `c'==1


set scheme plotplain

reg confIO c.q3##c.IOlr c.q17##c.IOgt $CONTROLS GE IN US 
estimates store `c'

margins, dydx(q3) at(IOlr=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of left-right orientation", size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "Left" 10 "Right") ///
	yline(0, lpattern(shortdash) lcolor(gray)) ///
	xtitle("Perceived IO profile", size(small)) name(`c'1) title("`c'")

margins, dydx(q17) at(IOgt=(0(1)10)) 
marginsplot, recast(line) ciopt(color(%20)) recastci(rarea) ///
	ytitle("Effect of GAL-TAN orientation", size(small)) ylabel(-.4(.1).4) yscale(titlegap(*-30)) ///
	xtick(0(1)10) xlabel(0 "GAL" 10 "TAN") ///
	yline(0, lpattern(shortdash) lcolor(gray))  ///
	xtitle("Perceived IO profile", size(small)) name(`c'2) title("`c'")
	
} 


esttab * using TableC13.rtf, se(%9.3f) b(%9.3f) ///
    ar2 scalars(N) varwidth(16) starlevels(* 0.05 ** 0.01 *** 0.001) replace

gr combine ///
	FullSample1 ///
	BR1 ///
	GE1 ///	
	IN1 ///
	US1 ///
	FullSample2 ///	
	BR2 ///
	GE2 ///	
	IN2 ///
	US2 ///
	, col(5) iscale(.8) imargin(0 3 0 6) 
graph export FigureC13.pdf, replace



********************************************************************************
** Table C14: Analysis of item non-response in the WVS7 
********************************************************************************

use "WVS_Cross-national_Wave_7_Stata_v2_0.dta", clear

rename Q88 confWHO
rename Q87 confWB
rename Q83 confUN
rename Q84 confIMF

recode confWHO  -5/-1=. 4=0 3=1 1=3
recode confIMF -5/-1=. 4=0 3=1 1=3
recode confUN -5/-1=. 4=0 3=1 1=3
recode confWB -5/-1=. 4=0 3=1 1=3


tab confIMF, missing
tab confIMF if C_COW_ALPHA=="INS", missing

tab confWB, missing
tab confWB if C_COW_ALPHA=="INS", missing

tab confWHO, missing
tab confWHO if C_COW_ALPHA=="INS", missing

tab confUN, missing
tab confUN if C_COW_ALPHA=="INS", missing


********************************************************************************
** Table C14: Analysis of item non-response in the WVS7 
********************************************************************************

use WVS_trends_3_0_reduced.dta, clear

tab E069_20
tab E069_20, nol
rename E069_20 confUN
recode confUN -5/-1=. 4=0 3=1 1=3

tab S002
tab confUN if S002==4, missing
tab confUN if COW_ALPHA=="INS"&S002==4, missing

tab confUN if S002==5, missing
tab confUN if COW_ALPHA=="INS"&S002==5, missing

tab confUN if S002==7, missing
tab confUN if COW_ALPHA=="INS"&S002==7, missing




********************************************************************************
********************************************************************************
** Appendix D Experimental Study
********************************************************************************
********************************************************************************


********************************************************************************
** Figure D1/2: Share of resps that correctly answered the manipulation checks
********************************************************************************

use Replication, replace
estimates drop _all
set scheme plotplain

gen mcheck1=(c_0034==q20)
gen mcheck2=(c_0035==q23)
gen mcheck3=(c_0036==q26)
gen mcheck4=(c_0037==q29)

replace mcheck1=. if c_0034==-66
replace mcheck2=. if c_0035==-66
replace mcheck3=. if c_0036==-66
replace mcheck4=. if c_0037==-66

replace mcheck1=100 if mcheck1==1
replace mcheck2=100 if mcheck2==1
replace mcheck3=100 if mcheck3==1
replace mcheck4=100 if mcheck4==1

label define treat 1 "GAL" 2 "TAN" 3 "Left" 4 "Right", replace
label value c_0034-c_0037 treat

sum mcheck*

** 
graph drop _all

graph bar (mean) mcheck1 if c_0034>0, over(c_0034) ylabel(0(20)100) /// 
	ytitle("%correct") nofill title("Migration", size(medium)) name(migration) 
graph bar (mean) mcheck2 if c_0035>0, over(c_0035) ylabel(0(20)100)  /// 
	ytitle("%correct") nofill title("Trade", size(medium)) name(trade)
graph bar (mean) mcheck3 if c_0036>0, over(c_0036) ylabel(0(20)100)  /// 
	ytitle("%correct") nofill title("Climate", size(medium)) name(climate)
graph bar (mean) mcheck4 if c_0037>0, over(c_0037) ylabel(0(20)100)  /// 
	ytitle("%correct") nofill title("Peace", size(medium)) name(peace)

gr combine migration trade climate peace, col(2)  imargin(0 0 0 0)
graph export FigureD1.pdf, replace   

** 

graph drop _all

graph bar (mean) mcheck1 if c_0034>0, over(country) ylabel(0(20)100)  /// 
	ytitle("%correct") nofill title("Migration", size(medium)) name(migration) 
graph bar (mean) mcheck2 if c_0035>0, over(country) ylabel(0(20)100)  /// 
	ytitle("%correct") nofill title("Trade", size(medium)) name(trade)
graph bar (mean) mcheck3 if c_0036>0, over(country) ylabel(0(20)100)  /// 
	ytitle("%correct") nofill title("Climate", size(medium)) name(climate)
graph bar (mean) mcheck4 if c_0037>0, over(country) ylabel(0(20)100)  /// 
	ytitle("%correct") nofill title("Peace", size(medium)) name(peace)

gr combine migration trade climate peace, col(2)  imargin(0 0 0 0)
graph export FigureD2.pdf, replace   


********************************************************************************
** Table D1: Balance tests
********************************************************************************

use Replication, replace
estimates drop _all
set scheme plotplain
graph drop _all

recode c_0034-c_0037 (-66=0)
eststo clear

reg 			age 		i.c_0034
est sto 	E1_age
testparm 		age 		i.c_0034
estadd scalar p_value = r(p)

reg 			male		i.c_0034
est sto 	E1_male
testparm 		male 		i.c_0034
estadd scalar p_value = r(p)

reg 			education	i.c_0034
est sto 	E1_education
testparm 		education 	i.c_0034
estadd scalar p_value = r(p)

reg 			q3 			i.c_0034
est sto 	E1_q3
testparm 		q3 			i.c_0034
estadd scalar p_value = r(p)

reg 			q17 		i.c_0034
est sto 	E1_q17
testparm 		q17 		i.c_0034
estadd scalar p_value = r(p)

**

reg 			age 		i.c_0035
est sto 	E2_age
testparm 		age 		i.c_0035
estadd scalar p_value = r(p)

reg 			male		i.c_0035
est sto 	E2_male
testparm 		male 		i.c_0035
estadd scalar p_value = r(p)

reg 			education	i.c_0035
est sto 	E2_education
testparm 		education 	i.c_0035
estadd scalar p_value = r(p)

reg 			q3 			i.c_0035
est sto 	E2_q3
testparm 		q3 			i.c_0035
estadd scalar p_value = r(p)

reg 			q17 		i.c_0035
est sto 	E2_q17
testparm 		q17 		i.c_0035
estadd scalar p_value = r(p)

**

reg 			age 		i.c_0036
est sto 	E3_age
testparm 		age 		i.c_0036
estadd scalar p_value = r(p)

reg 			male		i.c_0036
est sto 	E3_male
testparm 		male 		i.c_0036
estadd scalar p_value = r(p)

reg 			education	i.c_0036
est sto 	E3_education
testparm 		education 	i.c_0036
estadd scalar p_value = r(p)

reg 			q3 			i.c_0036
est sto 	E3_q3
testparm 		q3 			i.c_0036
estadd scalar p_value = r(p)

reg 			q17 		i.c_0036
est sto 	E3_q17
testparm 		q17 		i.c_0036
estadd scalar p_value = r(p)

**

reg 			age 		i.c_0037
est sto 	E4_age
testparm 		age 		i.c_0037
estadd scalar p_value = r(p)

reg 			male		i.c_0037
est sto 	E4_male
testparm 		male 		i.c_0037
estadd scalar p_value = r(p)

reg 			education	i.c_0037
est sto 	E4_education
testparm 		education 	i.c_0037
estadd scalar p_value = r(p)

reg 			q3 			i.c_0037
est sto 	E4_q3
testparm 		q3 			i.c_0037
estadd scalar p_value = r(p)

reg 			q17 		i.c_0037
est sto 	E4_q17
testparm 		q17 		i.c_0037
estadd scalar p_value = r(p)

esttab * using TableD1.rtf, se(%9.2f) b(%9.2f) ///
	scalar(F p_value) l replace unstack wide nobase

********************************************************************************
** Figure D3: Unconditional average treatment effects (treatment vs. control)
********************************************************************************

use Replication, replace
estimates drop _all
set scheme plotplain
graph drop _all

**Migration 

reg  q19 E1_1c   if c_0034==q20 | c_0034==-66 //GAL
estimates store E1_1c

reg  q19 E1_2c   if c_0034==q20 | c_0034==-66 //TAN
estimates store E1_2c

reg   q19 E1_3c   if c_0034==q20 | c_0034==-66 //L
estimates store E1_3c
 
reg   q19 E1_4c  if c_0034==q20 | c_0034==-66 //R
estimates store E1_4c

	
**Trade 

reg  q22 E2_1c  if c_0035==q23 | c_0035==-66 //GAL
estimates store E2_1c

reg  q22 E2_2c  if c_0035==q23 | c_0035==-66 //TAN
estimates store E2_2c

reg   q22 E2_3c   if c_0035==q23 | c_0035==-66 //L
estimates store E2_3c
 
reg   q22 E2_4c   if c_0035==q23 | c_0035==-66 //R
estimates store E2_4c
	
	
**Climate 

reg  q25 E3_1c  if c_0036==q26 | c_0036==-66 //GAL
estimates store E3_1c

reg  q25 E3_2c  if c_0036==q26 | c_0036==-66 //TAN
estimates store E3_2c

reg   q25 E3_3c   if c_0036==q26 | c_0036==-66 //L
estimates store E3_3c
 
reg   q25 E3_4c   if c_0036==q26 | c_0036==-66 //R
estimates store E3_4c

	
**Peacebuilding 

reg  q25 E4_1c  if c_0037==q29 | c_0037==-66 //GAL
estimates store E4_1c

reg  q25 E4_2c  if c_0037==q29 | c_0037==-66 //TAN
estimates store E4_2c

reg   q25 E4_3c   if c_0037==q29 | c_0037==-66 //L
estimates store E4_3c
 
reg   q25 E4_4c   if c_0037==q29 | c_0037==-66 //R
estimates store E4_4c

graph drop _all
set scheme plotplain
coefplot E1_1c E1_2c E1_3c E1_4c, ///
	keep(E1_*) xline(0) title(Migration) legend(off) name(m)
coefplot E2_1c E2_2c E2_3c E2_4c,  ///
	keep(E2_*) xline(0) title(Trade) legend(off) name(t)
coefplot E3_1c E3_2c E3_3c E3_4c,  ///
	keep(E3_*) xline(0) title(Climate) legend(off) name(c)
coefplot E4_1c E4_2c E4_3c E4_4c,  ///
	keep(E4_*) xline(0) title(Peace) legend(off) name(p)

gr combine m t c p, col(2)  imargin(0 0 0 0)
graph export FigureD3.pdf, replace

********************************************************************************
** Table D2: Marginal effect of ideological orientation by treatment group
********************************************************************************

** See code for Figure 5 above

********************************************************************************
** Table D3: 1-sample t-tests (2-t'd) of UN profile to equal middle cat (5)
********************************************************************************

use Replication, replace

asdoc ttest q39 ==5, save(TableD3a.rtf) replace 
asdoc bysort country: ttest q39 ==5, rowappend 

asdoc ttest q43 ==5, save(TableD3b.rtf) replace 
asdoc bysort country: ttest q43 ==5, rowappend 

********************************************************************************
** Table D4: 1-sample t-tests (2-t'd) of ave. IO profile to eq. middle cat (=5)
********************************************************************************

use Replication, replace

gen mIO_lr = (q39+ q47+ q55+ q63)/4
gen mIO_gt = (q43+ q51+ q59+ q67)/4	  

asdoc ttest mIO_lr ==5, save(TableD4a.rtf) replace
asdoc bysort country: ttest mIO_lr ==5, rowappend

asdoc ttest mIO_gt ==5, save(TableD4b.rtf) replace 
asdoc bysort country: ttest mIO_gt ==5, rowappend

********************************************************************************
** Table D5: Marginal effect of ideological orientation, weighted data
********************************************************************************

use Replication, replace
estimates drop _all
set scheme plotplain
graph drop _all

vl create CONTROLS = (intcoop confgov finsathousehold polsatisfaction ///
	feelcountry feelworld age male education GE IN US)


**Migration 

gen treat1 = E1_34c
label value treat1 m_treatmentLRc
reg   q19 c.q3##b1.treat1 q17 $CONTROLS  if c_0034==q20  | c_0034==-66 [pw=weight] 
estimates store E1_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2)) ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter)  ///
	name(E1_34c) title("Migration")

gen treat2=E1_12c
label value treat2 m_treatmentGTc
reg  q19 c.q17##b1.treat2 q3  $CONTROLS  if c_0034==q20 | c_0034==-66 [pw=weight] 
estimates store E1_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E1_12c) title("Migration")


**Trade 

replace treat1=E2_34c
label value treat1 m_treatmentLRc
reg   q22 c.q3##b1.treat1 q17 $CONTROLS  if c_0035==q23  | c_0035==-66 [pw=weight] 
estimates store E2_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E2_34c) title("Trade")

replace treat2=E2_12c
label value treat2 m_treatmentGTc
reg  q22 c.q17##b1.treat2 q3 $CONTROLS  if c_0035==q23 | c_0035==-66 [pw=weight] 
estimates store E2_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E2_12c) title("Trade")


**Climate 

replace treat1=E3_34c
label value treat1 m_treatmentLRc
reg  q25 c.q3##b1.treat1 q17 $CONTROLS  if c_0036==q26  | c_0036==-66 [pw=weight]
estimates store E3_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E3_34c) title("Climate")

replace treat2=E3_12c
label value treat2 m_treatmentGTc
reg  q25 c.q17##b1.treat2 q3 $CONTROLS  if c_0036==q26 | c_0036==-66 [pw=weight] 
estimates store E3_12c
margins, dydx(q17) at(treat2=(0(1)2))
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E3_12c) title("Climate")



**Peacebuilding

replace treat1=E4_34c
label value treat1 m_treatmentLRc
reg  q28 c.q3##b1.treat1 q17 $CONTROLS  if c_0037==q29  | c_0037==-66 [pw=weight]
estimates store E4_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E4_34c) title("Peace")

replace treat2=E4_12c
label value treat2 m_treatmentGTc
reg  q28 c.q17##b1.treat2 q3 $CONTROLS  if c_0037==q29 | c_0037==-66 [pw=weight] 
estimates store E4_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E4_12c) title("Peace")


esttab * using TableD5.rtf, se(%9.3f) b(%9.3f) l nobase ///
      scalars(N) ar2 varwidth(16) starlevels(  * 0.05 ** 0.01 *** 0.001) replace
gr combine ///
	E1_34c ///
	E2_34c ///
	E3_34c ///
	E4_34c ///
	E1_12c ///
	E2_12c ///	
	E3_12c ///
	E4_12c ///	
	, col(4) iscale(.8) imargin(0 3 0 6) 
graph export FigureD5.pdf, replace   


********************************************************************************
** Table D6: Marginal effect of ideological orientation, one dimension
********************************************************************************

use Replication, replace
estimates drop _all
set scheme plotplain
graph drop _all

vl create CONTROLS = (intcoop confgov finsathousehold polsatisfaction ///
	feelcountry feelworld age male education GE IN US)


**Migration 

gen treat1 = E1_34c
label value treat1 m_treatmentLRc
reg   q19 c.q3##b1.treat1  $CONTROLS   if c_0034==q20  | c_0034==-66 //LR
estimates store E1_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E1_34c) title("Migration")

gen treat2=E1_12c
label value treat2 m_treatmentGTc
reg  q19 c.q17##b1.treat2  $CONTROLS    if c_0034==q20 | c_0034==-66 //GAL-TAN
estimates store E1_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E1_12c) title("Migration")


**Trade 

replace treat1=E2_34c
label value treat1 m_treatmentLRc
reg   q22 c.q3##b1.treat1 $CONTROLS   if c_0035==q23  | c_0035==-66 //LR
estimates store E2_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E2_34c) title("Trade")

replace treat2=E2_12c
label value treat2 m_treatmentGTc
reg  q22 c.q17##b1.treat2  $CONTROLS   if c_0035==q23 | c_0035==-66 //GAL-TAN
estimates store E2_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E2_12c) title("Trade")


**Climate 

replace treat1=E3_34c
label value treat1 m_treatmentLRc
reg  q25 c.q3##b1.treat1  $CONTROLS   if c_0036==q26  | c_0036==-66 //LR
estimates store E3_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E3_34c) title("Climate")
	
replace treat2=E3_12c
label value treat2 m_treatmentGTc
reg  q25 c.q17##b1.treat2  $CONTROLS   if c_0036==q26 | c_0036==-66 //GAL-TAN
estimates store E3_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E3_12c) title("Climate")

	
		
**Peacebuilding 

replace treat1=E4_34c
label value treat1 m_treatmentLRc
reg  q28 c.q3##b1.treat1  $CONTROLS   if c_0037==q29  | c_0037==-66 //LR
estimates store E4_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E4_34c) title("Peace")

replace treat2=E4_12c
label value treat2 m_treatmentGTc
reg  q28 c.q17##b1.treat2  $CONTROLS   if c_0037==q29 | c_0037==-66 //GAL-TAN
estimates store E4_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E4_12c) title("Peace")


esttab * using TableD6.rtf, se(%9.3f) b(%9.3f) l nobase ///
      scalars(N) ar2 varwidth(16) starlevels(  * 0.05 ** 0.01 *** 0.001) replace
	  
gr combine ///
	E1_34c ///
	E2_34c ///
	E3_34c ///
	E4_34c ///
	E1_12c ///
	E2_12c ///	
	E3_12c ///
	E4_12c ///	
	, col(4) iscale(.8) imargin(0 3 0 6) 
graph export FigureD6.pdf, replace   



********************************************************************************
** Table D7: Marginal effect of ideological orientation, all respondents
********************************************************************************

use Replication, replace
estimates drop _all
set scheme plotplain
graph drop _all

vl create CONTROLS = (intcoop confgov finsathousehold polsatisfaction ///
	feelcountry feelworld age male education GE IN US)


**Migration 

gen treat1 = E1_34c
label value treat1 m_treatmentLRc
reg   q19 c.q3##b1.treat1 q17 $CONTROLS  
estimates store E1_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E1_34c) title("Migration")

gen treat2=E1_12c
label value treat2 m_treatmentGTc
reg  q19 c.q17##b1.treat2 q3  $CONTROLS  
estimates store E1_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E1_12c) title("Migration")


**Trade 

replace treat1=E2_34c
label value treat1 m_treatmentLRc
reg   q22 c.q3##b1.treat1 q17 $CONTROLS  
estimates store E2_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E2_34c) title("Trade")

replace treat2=E2_12c
label value treat2 m_treatmentGTc
reg  q22 c.q17##b1.treat2 q3 $CONTROLS  
estimates store E2_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E2_12c) title("Trade")


**Climate 

replace treat1=E3_34c
label value treat1 m_treatmentLRc
reg  q25 c.q3##b1.treat1 q17 $CONTROLS  
estimates store E3_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E3_34c) title("Climate")
	
replace treat2=E3_12c
label value treat2 m_treatmentGTc
reg  q25 c.q17##b1.treat2 q3 $CONTROLS  
estimates store E3_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E3_12c) title("Climate")

	
		
**Peacebuilding 

replace treat1=E4_34c
label value treat1 m_treatmentLRc
reg  q28 c.q3##b1.treat1 q17 $CONTROLS  
estimates store E4_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E4_34c) title("Peace")

replace treat2=E4_12c
label value treat2 m_treatmentGTc
reg  q28 c.q17##b1.treat2 q3 $CONTROLS  
estimates store E4_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E4_12c) title("Peace")

esttab * using TableD7.rtf, se(%9.3f) b(%9.3f) l nobase ///
      scalars(N) ar2 varwidth(16) starlevels(  * 0.05 ** 0.01 *** 0.001) replace	
	  
gr combine ///
	E1_34c ///
	E2_34c ///
	E3_34c ///
	E4_34c ///
	E1_12c ///
	E2_12c ///	
	E3_12c ///
	E4_12c ///	
	, col(4) iscale(.8) imargin(0 3 0 6) 
graph export FigureD7.pdf, replace   
	
	


********************************************************************************
** Table D8: Marginal effect of ideological orientation, countrywise results
********************************************************************************


foreach c in  BR GE IN US {

********

use Replication, replace
estimates drop _all
set scheme plotplain
graph drop _all

vl create CONTROLS = (intcoop confgov finsathousehold polsatisfaction ///
	feelcountry feelworld age male education)
	
keep if `c'==1

**Migration 

gen treat1 = E1_34c
label value treat1 m_treatmentLRc
reg   q19 c.q3##b1.treat1 q17 $CONTROLS  if c_0034==q20  | c_0034==-66 //LR
estimates store E1_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E1_34c) title("Migration")

gen treat2=E1_12c
label value treat2 m_treatmentGTc
reg  q19 c.q17##b1.treat2 q3  $CONTROLS  if c_0034==q20 | c_0034==-66 //GAL-TAN
estimates store E1_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E1_12c) title("Migration")


**Trade 

replace treat1=E2_34c
label value treat1 m_treatmentLRc
reg   q22 c.q3##b1.treat1 q17 $CONTROLS  if c_0035==q23  | c_0035==-66 //LR
estimates store E2_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E2_34c) title("Trade")

replace treat2=E2_12c
label value treat2 m_treatmentGTc
reg  q22 c.q17##b1.treat2 q3 $CONTROLS  if c_0035==q23 | c_0035==-66 //GAL-TAN
estimates store E2_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E2_12c) title("Trade")


**Climate 

replace treat1=E3_34c
label value treat1 m_treatmentLRc
reg  q25 c.q3##b1.treat1 q17 $CONTROLS  if c_0036==q26  | c_0036==-66 //LR
estimates store E3_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E3_34c) title("Climate")
	
replace treat2=E3_12c
label value treat2 m_treatmentGTc
reg  q25 c.q17##b1.treat2 q3 $CONTROLS  if c_0036==q26 | c_0036==-66 //GAL-TAN
estimates store E3_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E3_12c) title("Climate")

	
		
**Peacebuilding 

replace treat1=E4_34c
label value treat1 m_treatmentLRc
reg  q28 c.q3##b1.treat1 q17 $CONTROLS  if c_0037==q29  | c_0037==-66 //LR
estimates store E4_34c
margins, dydx(q3) at(treat1=(0(1)2))
marginsplot,  ///
	ytitle("Effect of left-right orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E4_34c) title("Peace")

replace treat2=E4_12c
label value treat2 m_treatmentGTc
reg  q28 c.q17##b1.treat2 q3 $CONTROLS  if c_0037==q29 | c_0037==-66 //GAL-TAN
estimates store E4_12c
margins, dydx(q17) at(treat2=(0(1)2)) 
marginsplot,  ///
	ytitle("Effect of GAL-TAN orientation",  ///
	size(small)) ylabel(-.2(.1).4) yscale(titlegap(*-05))  ///
	xtitle("") xscale(range(-.2 2.2))  ///
	yline(0, lpattern(shortdash) lcolor(gray)) recast(scatter) ///
	name(E4_12c) title("Peace")


esttab * using TableD_`c'.rtf, se(%9.3f) b(%9.3f) l nobase ///
      scalars(N) ar2 varwidth(16) starlevels(  * 0.05 ** 0.01 *** 0.001) replace
	  
gr combine ///
	E1_34c ///
	E2_34c ///
	E3_34c ///
	E4_34c ///
	E1_12c ///
	E2_12c ///	
	E3_12c ///
	E4_12c ///	
	, col(4) iscale(.8) imargin(0 3 0 6) 
graph export FigureD_`c'.pdf, replace   

******
} 



********************************************************************************
** Table D12: Marginal effect of ideological orientation (no control group)
********************************************************************************

use Replication, replace
estimates drop _all
set scheme plotplain
graph drop _all

vl create CONTROLS = (intcoop confgov finsathousehold polsatisfaction ///
	feelcountry feelworld age male education GE IN US)

	
**Migration 

gen treat1=E1_34c
label value treat1 m_treatmentLRc
reg   q19 c.q3##treat1 q17 $CONTROLS     if c_0034==q20   //LR
estimates store E1_34c


gen treat2=E1_12c
label value treat2 m_treatmentGTc
reg  q19 c.q17##treat2 q3  $CONTROLS  if c_0034==q20  //GAL-TAN
estimates store E1_12c



**Trade 

replace treat1=E2_34c
label value treat1 m_treatmentLRc
reg   q22 c.q3##treat1 q17 $CONTROLS  if c_0035==q23   //LR
estimates store E2_34c


replace treat2=E2_12c
label value treat2 m_treatmentGTc
reg  q22 c.q17##treat2 q3 $CONTROLS   if c_0035==q23  //GAL-TAN
estimates store E2_12c



**Climate 

replace treat1=E3_34c
label value treat1 m_treatmentLRc
reg  q25 c.q3##treat1 q17 $CONTROLS   if c_0036==q26   //LR
estimates store E3_34c

	
replace treat2=E3_12c
label value treat2 m_treatmentGTc
reg  q25 c.q17##treat2 q3 $CONTROLS   if c_0036==q26  //GAL-TAN
estimates store E3_12c


	
		
**Peacebuilding 

replace treat1=E4_34c
label value treat1 m_treatmentLRc
reg  q28 c.q3##treat1 q17 $CONTROLS   if c_0037==q29  //LR
estimates store E4_34c


replace treat2=E4_12c
label value treat2 m_treatmentGTc
reg  q28 c.q17##treat2 q3 $CONTROLS   if c_0037==q29  //GAL-TAN
estimates store E4_12c


esttab * using TableD12.rtf, se(%9.3f) b(%9.3f) l nobase ///
      scalars(N) ar2 varwidth(16) starlevels(  * 0.05 ** 0.01 *** 0.001) replace


********************************************************************************
********************************************************************************
*************************** DONE :) ********************************************
********************************************************************************
********************************************************************************

